<?php

# Dependency: class MySQL
# 			  class SourceQuery
#			  $_SESSION['config']

use xPaw\SourceQuery\SourceQuery;

# Gameserver Class
class LiFServer {
	
	// Functional properties
	private $db, $sq;
	
	// Data properties
	protected $ip, $port, $ispublic;
	protected $ttmod_version = NULL;
	public $serverinfo   = NULL;
	public $serverrules  = NULL;
	
	// Constructor
	public function __construct( $ip, $port, $enable_query ) {
		
		$this->ip = $ip;
		$this->port = $port;
		$this->ispublic = $enable_query;
		
	}
	
	// Connect to gameserver database
	public function db_connect() {
		
		if( ! $this->db ) $this->db = new MySQL( $_SESSION['config']['db_user'], $_SESSION['config']['db_pass'], $_SESSION['config']['db_name'], $_SESSION['config']['db_ip'], intval($_SESSION['config']['db_port']), 'utf8' );
		
		return TRUE;
		
	}
	
	// Connect to server via sourcequery library 
	public function sq_connect() {
		
		$this->sq = new SourceQuery();
		$this->sq->Connect( $this->ip, $this->port + 2, 2, SourceQuery::SOURCE );
		
	}
	
	// Is server mod installed?
	public function detect_servermod() {
		
		$this->db_connect();
		
		if( $this->ttmod_version === NULL ) $this->ttmod_version = $this->get_servermod_version();
			
		return ( $this->ttmod_version > 0 );
		
	}
	
	// Get TTmod version
	public function get_servermod_version() {
		
		$this->db_connect();
		
		if( $this->ttmod_version !== NULL ) return $this->ttmod_version;
		
		if( ! $this->db->table_exists('nyu_ttmod_info') ) {
			
			// Detect old versions
			if( $this->db->table_exists('nyu_online_players') ) $this->ttmod_version = 1;
			else $this->ttmod_version = 0;
				
		} else {
		
			// Get from info table
			$info = $this->db->query( "SELECT ttmod_version FROM nyu_ttmod_info LIMIT 1", FALSE );
			$this->ttmod_version = floatval($info['ttmod_version']);
		
		}
		
		return $this->ttmod_version;
		
	}
	
	// Get all guild claims from database
	public function get_claims() {
		
		$this->db_connect();
		
		$claims = $this->db->query( "SELECT gl.*, ( gl.CenterGeoID >> 18 ) AS TerID, ( gl.CenterGeoID & ((1 << 9) - 1) ) AS `x`, ( (gl.CenterGeoID >> 9) & ((1 << (9)) - 1) ) AS `y`, g.Name AS guild, g.CreateTimestamp AS ctime, g.GuildCharter, g.GuildTag
									 FROM guild_lands AS gl 
									 LEFT JOIN guilds g ON gl.GuildID = g.ID
									 WHERE LandType < 4
									 ORDER BY guild" );
		
		return $claims;
		
	}
	
	// Get outposts
	public function get_outposts() {
		
		$this->db_connect();
		
		$outposts = array();
		if( $this->db->table_exists('outposts') ) {
			$outposts = $this->db->query( "SELECT o.ID, uo.GeoDataID, ot.Name AS BuildingName, g.Name AS GuildName, uo.ObjectTypeID
											FROM outposts o
											JOIN unmovable_objects uo ON o.UnmovableObjectID = uo.ID
											JOIN objects_types ot ON uo.ObjectTypeID = ot.ID
											LEFT JOIN guilds g ON o.OwnerGuildID = g.ID" );
		}
		
		return $outposts;
		
	}
	
	// Get all personal claims from database
	public function get_personal_claims() {
		
		$this->db_connect();
		
		$claims = $this->db->query( "SELECT c.Name, c.LastName, pl.CharID, pl.ID,
										( pl.GeoID2 >> 18 ) AS TerID1, ( pl.GeoID2 & ((1 << 9) - 1) ) AS x1, ( (pl.GeoID2 >> 9) & ((1 << (9)) - 1) ) AS y1,
										( pl.GeoID1 >> 18 ) AS TerID2, ( pl.GeoID1 & ((1 << 9) - 1) ) AS x2, ( (pl.GeoID1 >> 9) & ((1 << (9)) - 1) ) AS y2
									 FROM `character` c, `personal_lands` pl
									 WHERE pl.IsTemp = 0 AND pl.CharID = c.ID" );
									 
		return $claims;
		
	}
	
	// Get all personal claims from database
	public function get_admin_lands() {
		
		$this->db_connect();
		
		$claims = $this->db->query( "SELECT ID, Name, 
										( GeoID2 >> 18 ) AS TerID1, ( GeoID2 & ((1 << 9) - 1) ) AS x1, ( (GeoID2 >> 9) & ((1 << (9)) - 1) ) AS y1,
										( GeoID1 >> 18 ) AS TerID2, ( GeoID1 & ((1 << 9) - 1) ) AS x2, ( (GeoID1 >> 9) & ((1 << (9)) - 1) ) AS y2
									 FROM admin_lands" );
									 
		return $claims;
		
	}
	
	// Get all guilds from database
	public function get_guilds() {
		
		$this->db_connect();
		
		return $this->db->query( "SELECT * FROM guilds ORDER BY Name" );

	}
	
	// Get guild standings
	public function get_guild_standings( $guild1 = NULL, $guild2 = NULL ) {
		
		$this->db_connect();
		
		$add1 = is_null($guild1) ? "" : "AND GuildID1 = '$guild1'";
		$add2 = is_null($guild2) ? "" : "AND GuildID2 = '$guild2'";
		
		// All standings that aren't default (1)
		return $this->db->query( "SELECT GuildID1, GuildID2, StandingTypeID FROM guild_standings WHERE StandingTypeID <> 1 $add1 $add2" );
		
	}
	
	// Get all characters from database
	public function get_characters() {
		
		$this->db_connect();
		
		$characters = $this->db->query( "SELECT c.ID AS CharID, TRIM('\r\n' FROM TRIM(c.Name)) AS FirstName, TRIM('\r\n' FROM TRIM(c.LastName)) AS LastName, c.GuildID, c.GuildRoleId, c.AccountID, ASCII( SUBSTRING(c.appearance,1) ) AS gender, a.SteamID
										 FROM `character` c,  `account` a
										 WHERE a.ID = c.AccountID
										 ORDER BY FirstName ASC" );
		
		return $characters;
		
	}
	
	// Get extended list of all characters
	public function get_characters_list() {
		
		if( $this->get_servermod_version() >= 1.3 ) {

			$query = "SELECT
						a.SteamID, a.IsActive AS AccountActive, a.ID AS AccountID, a.IsGM,
						c.ID, c.GeoID, c.GuildRoleID, c.IsActive AS CharActive, ROUND(c.Alignment/1000000) AS Alignment,
						c.CreateTimestamp AS CreateString, UNIX_TIMESTAMP(c.CreateTimestamp) AS CreateTimestamp,
						c.Name AS FirstName, c.LastName, CONCAT(c.Name, ' ', c.LastName) AS Name,
						ASCII( SUBSTRING(c.appearance,1) ) AS Gender,
						g.ID AS GuildID, g.Name AS GuildName,
						COUNT(op.CharID) AS is_online,
						MAX(`ts`.`time`) AS LastOnlineString,
						UNIX_TIMESTAMP(MAX(`ts`.`time`)) AS LastOnlineTimestamp,
						ROUND( COUNT(`ts`.`ID`) * 5 / 60 ) AS Playtime,
						ci.deaths AS Deaths, ci.kills AS Kills
					  FROM
						`account` a, `character` c
					  LEFT JOIN `nyu_chars_info` ci
					    ON c.ID = ci.CharID
					  LEFT JOIN `guilds` g
					    ON c.GuildID = g.ID
					  LEFT JOIN `nyu_ttmod_tokens` op
						ON c.ID = op.CharID
					  LEFT JOIN `nyu_tracker_chars` tc
						  INNER JOIN `nyu_tracker_stats` ts
						  ON ts.ID = tc.stat_id
						ON tc.CharacterID = c.ID
					  WHERE a.ID = c.AccountID
					  GROUP BY c.ID
					  ORDER BY c.Name";
	  
		} else {
	  
			$query = "SELECT
						a.SteamID, a.IsActive AS AccountActive, a.ID AS AccountID, a.IsGM,
						c.ID, c.GeoID, c.GuildRoleID, c.CreateTimestamp, c.IsActive AS CharActive, ROUND(c.Alignment/1000000) AS Alignment, 
						c.CreateTimestamp AS CreateString, UNIX_TIMESTAMP(c.CreateTimestamp) AS CreateTimestamp,
						c.Name AS FirstName, c.LastName, CONCAT(c.Name, ' ', c.LastName) AS Name,
						ASCII( SUBSTRING(c.appearance,1) ) AS Gender,
						g.ID AS GuildID, g.Name AS GuildName,
						0 AS is_online
					  FROM
						`account` a, `character` c
					  LEFT JOIN `guilds` g
					    ON c.GuildID = g.ID
					  WHERE a.ID = c.AccountID
					  GROUP BY c.ID
					  ORDER BY c.Name";
			
		}
		
		return $this->db->query($query);
		
	}
	
	// Get permanent server GM characters
	public function get_gm_characters() {
		
		$this->db_connect();
		
		return $this->db->query( "SELECT c.ID, c.Name, c.LastName FROM `character` c, `account` a WHERE c.AccountID = a.ID AND a.isGM = 1" );
		
	}
	
	// Check for yo version 1.3.6.0
	public function is_yo136() {
		
		$this->db_connect();
		
		return ! $this->db->table_exists("outposts");
		
	}
	
	// Get unix timestamps of next scheduled restarts
	public function get_restart_timestamps( $input ) {
		
		$restarts = explode(' ', $input);
		
		// Convert each timestring to unix timestamp
		foreach( $restarts AS &$time ) {
			
			$hres = intval( substr($time, 0, 2) );
			$ires = intval( substr($time, -2) );
			
			$mktime = mktime($hres, $ires, 0);
			if( $mktime < time() ) $mktime = $mktime + 24*60*60;

			$time = $mktime;
			
		}
		
		// Add a second one if we have less than two.
		if( count($restarts) < 2 ) {
			array_push( $restarts, ($mktime+24*60*60) );
		}
		
		return $restarts;
		
	}
	
	// Get Judgement Hour schedule
	public function get_judgement_hour() {
		
		// Return empty array if server is private. Can't detect JH values in this case
		if( ! $this->ispublic ) return array();
		
		// Connect sourcequery socket if necessary
		if( ! $this->sq ) $this->sq_connect(); 
		
		// In case of any problems, will return an empty array instead of throwing an exception
		try {
			
			// Get 'rules' array from gameserver
			$this->serverrules = $this->sq->GetRules();

			// Return empty array if disabled
			if( empty($this->serverrules['weekSchedule']) || ! $this->serverrules['duration'] ) return array();
			
			// Convert the bullshit returned from SourceQuery to an offset
			$tz_utc_offset = date('Z') - date('I')*3600;	// Real offset seconds minus DST seconds cause Russians don't like DST
			
			// Create an array of timestamps for the next JH events
			$days = explode( ' ', $this->serverrules['weekSchedule'] );
			$timestamps = array();
			foreach( $days AS $weekday ) $timestamps[] = strtotime("$weekday {$this->serverrules['startTime']}") - $tz_utc_offset;

			// Add a second one if it's only one timestamp
			if( count($timestamps) < 2 ) $timestamps[] = $timestamps[0] + (60*60*24*7);

			// Return timestamps and duration
			return array( 'timestamps' => $timestamps, 'duration' => intval($this->serverrules['duration']) );
			
		// Handle error if no communication with server possible
		} catch( Exception $e ) {

			return array();
			
		}
		
	}
	
	// Get a specifc detail of the server
	public function get_server_info( $key ) {
		
		switch( $key ) {
			
			// Online Player Count
			case 'Players':
				if( $this->ispublic ) {
					// Query server if necessary
					if( $this->serverinfo === NULL ) $this->query_server();
					// Return player count
					if( isSet($this->serverinfo[$key]) ) return intval($this->serverinfo[$key]);
					// If server is offline
					else return FALSE;
				// If server query disabled, but mod is installed, count online players in database. Anyway, this means we won't get maximum player count (server slots).
				} elseif( $this->detect_servermod() ) return $this->get_db_online_count();	
				// If both disabled, this function is never called. Actually... or is it..
				else return FALSE;
			break;
			
			// Server Slots
			case 'MaxPlayers':
				if( ! $this->ispublic ) return FALSE;
				// Query server if necessary
				if( $this->serverinfo === NULL ) $this->query_server();
				// Return the requested info
				if( isSet($this->serverinfo[$key]) ) return intval($this->serverinfo[$key]);
				// If server is offline
				else return FALSE;
			break;
				
		}
				
	}
	
	// Get number of online players from database
	public function get_db_online_count() {
		
		$this->db_connect();
		if( $this->get_servermod_version() >= 1.3 ) {
			$result = $this->db->query( "SELECT COUNT(CharID) AS num FROM nyu_ttmod_tokens", FALSE );
		} else {
			$result = $this->db->query( "SELECT COUNT(CharacterID) AS num FROM nyu_online_players", FALSE );
		}
		return intval($result['num']);

	}
	
	// Get online player names and, optionally, their positions
	public function get_online_players( $getpos ) {
		
		// Does not work without servermod. Return empty array if called.
		if( ! $this->detect_servermod() ) return array();
		
		$reference_table = $this->get_servermod_version() >= 1.3 ? 'nyu_ttmod_tokens' : 'nyu_online_players';
		$reference_cell  = $this->get_servermod_version() >= 1.3 ? 'CharId' : 'CharacterID';
		
		$result = $this->db->query( "SELECT 
										ASCII( SUBSTRING(c.appearance,1) ) AS gender,
										( c.GeoID >> 18 ) AS TerID,
										( c.GeoID & ((1 << 9) - 1) ) AS `x`,
										( (c.GeoID >> 9) & ((1 << (9)) - 1) ) AS `y`, 
										TRIM('\r\n' FROM TRIM(c.Name)) AS FirstName, 
										TRIM('\r\n' FROM TRIM(c.LastName)) AS LastName,
										c.ID
									 FROM `character` c, `$reference_table` op
									 WHERE op.$reference_cell = c.id
									 ORDER BY c.Name ASC" );
								
		foreach( $result AS &$player ) {
			
			// Convert to pixel coordinates
			if( $getpos ) {
				
				$coords = Livemap::terpos2pixelpos( $player['TerID'], $player['x'], $player['y'] );
				unset($player['TerID']);
				$player['x'] = $coords[0];
				$player['y'] = $coords[1];
				
			// Delte position information if we don't want it
			} else {
				
				unset($player['TerID']);
				unset($player['x']);
				unset($player['y']);
				
			}
		
			// Add tags
			$player['tags'] = array();
			foreach( $_SESSION['groups'] AS $group ) {
				$members = str_getcsv($group['members_csv']);
				// GM group special treatment
				if( $group['type_id'] === '2' ) {
					$gms = $this->get_gm_characters();
					foreach( $gms AS $gm ) array_push($members, $gm['ID']);
				}
				if( in_array($player['ID'], $members) ) {
					$player['tags'][] = array( 
						'tag' => $group['tag'],
						'tag_name' => $group['name'],
						'tag_color' => $group['tag_color']
					); 
				}
			}

		}
		
		return $result;
		
	}
	
	// Get all structures
	public function get_structures() {
		
		$this->db_connect();
		
		// Get all structures
		$buildings = $this->db->query( "SELECT ( uo.GeoDataID >> 18 ) AS TerID, ( uo.GeoDataID & ((1 << 9) - 1) ) AS `x`, ( (uo.GeoDataID >> 9) & ((1 << (9)) - 1) ) AS `y` FROM unmovable_objects uo WHERE uo.IsComplete = 1" );

		// Get pixel position for each building
		$result = array();
		foreach( $buildings AS $b ) $result[] = Livemap::terpos2pixelpos( $b['TerID'], $b['x'], $b['y'] );
		unset($buildings);
			
		return $result;
		
	}
	
	// Get all trading posts
	public function get_tradeposts() {
		
		$this->db_connect();
		
		return $this->db->query( "SELECT ID, GeoDataID FROM unmovable_objects WHERE IsComplete = 1 AND ObjectTypeID = 1077" );
		
	}
	
	// Get all roads
	public function get_paved_tiles( $use_cache = TRUE ) {
		
		$this->db_connect();
		
		// Prepare result array
		$result = array( 'stone' => [], 'marble' => [], 'slate' => [] );
		
		if( ! $use_cache || ! $this->db->table_exists('nyu_terrain_cache') ) {
			
			$rs = $this->db->query( "SELECT gp.Substance, gp.GeoDataID FROM geo_patch gp 
									 INNER JOIN ( SELECT GeoDataID, MAX(Version) MaxVer FROM geo_patch GROUP BY GeoDataID ) tmp ON tmp.GeoDataID = gp.GeoDataID AND tmp.MaxVer = gp.Version
									 WHERE gp.Substance IN (177, 180, 181)" );
									 
		} else {
			
			$rs = $this->db->query( "SELECT * FROM nyu_terrain_cache" );

		}
								 
		foreach( $rs AS &$tile ) {
			switch($tile['Substance']) {
				case '177':	$result['stone'][]  = Livemap::geoid2pixelpos( intval($tile['GeoDataID']) );	break;
				case '180': $result['marble'][] = Livemap::geoid2pixelpos( intval($tile['GeoDataID']) );	break;
				case '181': $result['slate'][]  = Livemap::geoid2pixelpos( intval($tile['GeoDataID']) );	break;
			}
			unset($tile);
		}
		
		return $result;
		
	}
	
	// Cache roads to dedicated table - return cached version sum
	public function cache_paved_tiles() {
	
		$this->db_connect();
		
		if( ! $this->db->table_exists('nyu_terrain_cache') ) $this->db->query( "CREATE TABLE IF NOT EXISTS `nyu_terrain_cache` ( `GeoDataID` INT(10) UNSIGNED NOT NULL, `Substance` TINYINT(3) UNSIGNED NOT NULL, PRIMARY KEY (`GeoDataID`) )" );

		$this->db->begin_transaction();
		
		$this->db->query( "DELETE FROM nyu_terrain_cache");
		$this->db->query( "INSERT IGNORE INTO nyu_terrain_cache
							SELECT gp.GeoDataID, gp.Substance FROM geo_patch gp 
							INNER JOIN ( SELECT GeoDataID, MAX(Version) MaxVer FROM geo_patch GROUP BY GeoDataID ) tmp 
								ON tmp.GeoDataID = gp.GeoDataID AND tmp.MaxVer = gp.Version
							WHERE gp.Substance IN (177, 180, 181)");
		$this->db->commit();

		return $this->get_geo_version_sum();
							
	}
	
	public function get_geo_version_sum() {
		
		$this->db_connect();
		
		$rs = $this->db->query( "SELECT SUM(GeoVersion) AS ver_sum FROM terrain_blocks", FALSE );
		
		return intval($rs['ver_sum']);
		
	}
	
	// Get current day of the year
	public function get_day( $daycycle ) {
		
		$base = 1404172800;
		$diff = time() - $base;
		$add  = round($diff * (24/$daycycle));
		
		$date = new DateTime();
		$date->setTimestamp( $base );
		$date->setTimezone( new DateTimeZone("UTC") );
		$date->modify( "+$add seconds" );
		$date->modify( "+12 hours" );

		return intval( $date->format('z') );
		
	}
	
	// Gets weather from $first to $last day of the year
	public function get_weather_array( $first, $last ) {
		
		$custompath = "weather/{$_SESSION['config']['ID']}_weather.xml";
		$xmlfile = file_exists($custompath) ? $custompath : 'weather/cm_weather1.xml';
		
		$xml = simplexml_load_file($xmlfile);
		
		$array = array();
		
		// Loop from $first to $last day of the year
		for( $i = $first; $i <= $last; $i++ ) {
			$day = $i < 365 ? $i : $i - 365;
			// Grab config from weather XML
			if( $day !== intval($xml->day[$day]['id']) ) throw new Exception('Invalid cm_weather1.xml file');
			// Translate to locale code
			switch( $xml->day[$day]->__toString() ) {
				case 'Shower': $weather = 'wrain';	break;
				case 'Cloudy': $weather = 'wclou';	break;
				case 'Snowy':  $weather = 'wsnow';	break;
				default: 	   $weather = 'wfair';	break;
			}
			$array[] = array( 'day' => $day, 'weather' => $weather, 'season' => $xml->day[$day]['season'] );
		}
		
		return $array;
		
	}
	
	// Get steamid from token
	public function get_token_details( $token ) {
		
		if( $this->get_servermod_version() < 1.2 ) return FALSE;
		
		$rs = $this->db->query( "SELECT a.SteamID, t.CharID
			FROM `nyu_ttmod_tokens` t, `character` c, `account` a 
			WHERE t.Token = '$token' AND t.CharID = c.ID AND c.AccountID = a.ID", FALSE );
			
		if( ! $rs ) return FALSE;
		
		return $rs;
		
	}
	
	// Get characters by steam ID
	public function get_steam_characters( $steam_id ) {
		
		if( ! $steam_id ) return array();
		
		$this->db_connect();
		
		$steam_id = $this->db->esc($steam_id);
		
		$query = "SELECT 
					c.ID, c.Name, c.LastName, ASCII( SUBSTRING(c.appearance,1) ) AS gender,
					c.GuildID, c.GuildRoleID, g.Name AS GuildName, gl.Radius
				  FROM `account` a
				  JOIN `character` c ON a.ID = c.AccountID
				  LEFT JOIN guilds g ON c.GuildID = g.ID
				  LEFT JOIN guild_lands gl ON gl.GuildID = g.ID AND gl.LandType < 4
				  WHERE c.AccountID = a.ID AND a.SteamID = '$steam_id'";
		
		return $this->db->query( $query );
		
	}
	
	// Get info from character ID
	public function get_character_info( $id ) {
		
		// Return empty array if no input
		if( ! $id ) return array();
		
		$this->db_connect();
		
		$query = "SELECT ID, Name, LastName FROM `character` WHERE ID IN ( $id )";
		
		return $this->db->query( $query );
		
	}
	
	// Get online stats from character
	public function get_character_stats( $id ) {
		
		if( ! $id = intval($id) ) return array();
		if( ! $this->detect_servermod() ) return array();
		
		$this->db_connect();
		
		$rs = $this->db->query( "SELECT HOUR(ts.time) AS hour, ROUND( COUNT(1) / (SELECT SUM(1) FROM nyu_tracker_chars WHERE CharacterID = tc.CharacterID) * 100, 1) AS share
			FROM nyu_tracker_stats ts, nyu_tracker_chars tc
			WHERE ts.ID = tc.stat_id AND tc.CharacterID = $id
			GROUP BY hour ORDER BY hour" );
		
		$output = array();
		foreach( $rs AS $row ) {
			while( count($output) < intval($row['hour']) ) $output[] = array( 'hour' => count($output), 'share' => 0.0 );
			$output[] = array( 'hour' => intval($row['hour']), 'share' => floatval($row['share']) );
		}
		while( count($output) < 24 ) $output[] = array( 'hour' => count($output), 'share' => 0.0 );
		
		return $output;
		
	}
	
	// Get charcter skills tree
	public function get_character_skills( $id ) {
		
		$this->db_connect();
		
		$skills = $this->db->query( "SELECT st.ID AS ID, st.Parent, st.Name, round(s.SkillAmount / 10000000, 2) AS Skill 
										FROM skill_type st LEFT JOIN skills s ON st.ID = s.SkillTypeID AND s.CharacterID = '$id'
										ORDER BY st.Name" );
		if( ! $skills ) return "{}";
		
		return $skills;
		
	}
	
	// Set charcter skill
	public function set_character_skill( $char_id, $skill_id, $skill_value ) {
		
		$skill_value = $skill_value * 10000000;
		
		$this->db_connect();
		
		$this->db->query( "INSERT INTO skills (CharacterID, SkillTypeID, SkillAmount) VALUES ('$char_id', '$skill_id', '$skill_value') ON DUPLICATE KEY UPDATE SkillAmount = '$skill_value'" );
		
		return TRUE;
		
	}
	
	// Get charcter inventory
	public function get_character_inventory( $id ) {
		
		$output = array( 'inventory' => array(), 'equipment' => array() );
		
		if( ! $id = intval($id) ) return $output;
		
		$this->db_connect();
		
		if( ! $char = $this->db->query( "SELECT RootContainerID, EquipmentContainerID FROM `character` WHERE ID = '$id'", FALSE ) ) return $output;
		
		// Equipped Items
		$output['equipment'] = $this->db->query( "SELECT ot.Name, i.Quality, i.Quantity, i.Durability, i.CreatedDurability, es.Slot 
													FROM `items` i, `equipment_slots` es, `objects_types` ot 
													WHERE i.ID = es.ItemID AND i.ObjectTypeId = ot.ID AND es.CharacterID = '$id'" );
		// Inventory Bag
		$output['inventory'] = $this->get_container_tree($char['RootContainerID']);

		return $output;
		
	}
	
	// Get content of a container
	public function get_container_tree( $id ) {
		
		$items = $this->db->query( "SELECT ot.Name, i.Quality, i.Quantity, i.Durability, i.CreatedDurability FROM `items` i, `objects_types` ot WHERE i.ContainerID = '$id' AND i.ObjectTypeId = ot.ID ORDER BY ot.Name" );
		
		$containers = $this->db->query( "SELECT c.ID, c.Quality, ot.Name FROM `containers` c, `objects_types` ot WHERE c.ParentID = '$id' AND c.ObjectTypeId = ot.ID ORDER BY ot.Name" );
									
		foreach( $containers AS $container ) {
			
			$container['content'] = $this->get_container_tree($container['ID']);
			array_push($items, $container);
			
		}
		
		return $items;
		
	}
	
	// Get list of all items on server
	public function get_all_items() {
		
		$this->db_connect();
		
		$query = "SELECT ID, Name FROM objects_types WHERE IsUnmovableobject = 0 AND IsMovableobject = 0 AND ParentID IS NOT NULL AND ID NOT IN (SELECT DISTINCT ParentID FROM objects_types WHERE ParentID IS NOT NULL) ORDER BY Name";
		
		return $this->db->query( $query );
		
	}
	
	// Add RCON command to queue
	public function add_rcon_command($cmd, $param1 = '', $param2 = '', $detail = '', $minutes = 0) {
		
		if( ! $this->detect_servermod() ) return FALSE;

		$detail = $this->db->esc($detail);
		
		$rcon_id = $this->db->query( "INSERT INTO nyu_rcon_queue (command, param1, param2, detail, exec_time) VALUES ('$cmd', '$param1', '$param2', '$detail', DATE_ADD(NOW(), INTERVAL $minutes MINUTE))" );
		
		return (bool)$rcon_id;

	}
	
	// Ban a player by character_id
	public function ban_player( $char_id ) {
		
		$this->db_connect();
		
		$this->db->query( "UPDATE `account` a, `character` c SET a.isActive = 0 WHERE a.ID = c.AccountID AND c.ID = '$char_id'" );
		
		return TRUE;
		
	}
	
	// Ban or unban a player by account_id
	public function set_account_status( $account_id, $active ) {
		
		$this->db_connect();
		
		$this->db->query( "UPDATE `account` SET isActive = " . intval($active) . " WHERE ID = '$account_id'" );
		
		return TRUE;
	}
	
	// Delete a character from database
	public function delete_character($char_id) {
		
		$this->db_connect();
		
		$result = $this->db->query("SELECT AccountID, GuildRoleID FROM `character` WHERE ID = '$char_id'", FALSE);
		
		if( ! $result || intval($result['GuildRoleID']) === 1 ) return FALSE;
		
		$this->db->query("CALL p_deleteCharacter($char_id, {$result['AccountID']})");

		return TRUE;
		
	}
	
	// Pass a query through
	public function passthru_db_query($query, $multi = TRUE) {
		
		$this->db_connect();
		
		return $this->db->query($query, $multi);
		
	}
	
	// Get stats/facts about the server
	public function get_stat_info( $what ) {
		
		$this->db_connect();
		
		switch( $what ) {
			
			case 'chars_total':
				$r = $this->db->query( "SELECT COUNT(ID) AS total_chars FROM `character`", FALSE );
				return intval($r['total_chars']);
			break;
				
			case 'tracker_first':
				if( ! $this->detect_servermod() ) return 'Never. Server mod is not installed.';
				$r = $this->db->query( "SELECT MIN(`time`) AS tracker_first FROM nyu_tracker_stats", FALSE );
				return $r['tracker_first'];
			break;
			
			case 'tracker_last':
				if( ! $this->detect_servermod() ) return 'Never. Server mod is not installed.';
				$r = $this->db->query( "SELECT MAX(`time`) AS tracker_last FROM nyu_tracker_stats", FALSE );
				return $r['tracker_last'];
			break;
			
		}
		
	}
	
	// Get char permissions on guild claims
	public function get_char_permissions( $char_id, $guild_id ) {
		
		$this->db_connect();
		
		// Grab source character info
		$char = $this->db->query( "SELECT ID, GuildID, GuildRoleID FROM `character` WHERE ID = '$char_id'", FALSE );
		// Get target guild info
		$guild = $this->db->query( "SELECT c.ID AS ClaimID, c.GuildLandID FROM `claims` c, `guild_lands` gl WHERE c.GuildLandID = gl.ID AND gl.LandType < 4 AND gl.GuildID = '$guild_id'", FALSE );
		
		if( ! $char || ! $guild ) return FALSE;
		
		$permissions = array();
		
		// is character guild member of owner guild?
		if( intval($char['GuildID']) === $guild_id ) {
			// Apply permissions for own rank
			$permissions[] = $this->db->query( 
				"SELECT cr.CanEnter, cr.CanBuild, cr.CanClaim, cr.CanUse, cr.CanDestroy 
					FROM claim_subjects cs
					INNER JOIN claim_rules cr ON cr.ClaimSubjectID = cs.ID AND cr.ClaimID = '{$guild['ClaimID']}'
					WHERE cs.GuildRoleID = '{$char['GuildRoleID']}'", FALSE
			);
		}

		// is character member in some other guild?
		if( $char['GuildID'] ) {
			// is some standing set?
			$std = $this->db->query( "SELECT StandingTypeID FROM `guild_standings` WHERE GuildID1 = '$guild_id' AND GuildID2 = '{$char['GuildID']}'", FALSE );
			if( $std ) {
				// Grab standing permissions
				$permissions[] = $this->db->query( "SELECT cr.CanEnter, cr.CanBuild, cr.CanClaim, cr.CanUse, cr.CanDestroy
															FROM claim_subjects cs
															INNER JOIN claim_rules cr ON cr.ClaimSubjectID = cs.ID AND cr.ClaimID = '{$guild['ClaimID']}'
															WHERE cs.StandingTypeID = '{$std['StandingTypeID']}'", FALSE );
			}
			// is some special shit set for the guild of this char?
			$permissions[] = $this->db->query( "SELECT cr.CanEnter, cr.CanBuild, cr.CanClaim, cr.CanUse, cr.CanDestroy
															FROM claim_rules cr, claim_subjects cs
															WHERE cr.ClaimID = '{$guild['ClaimID']}' AND cr.ClaimSubjectID = cs.ID AND cs.GuildID = '{$char['GuildID']}'", FALSE );
		}
		
		// is some special privilege set for this character?
		$permissions[] = $this->db->query( "SELECT cr.CanEnter, cr.CanBuild, cr.CanClaim, cr.CanUse, cr.CanDestroy
														FROM `claim_rules` cr, `claim_subjects` cs
														WHERE cr.ClaimID = '{$guild['ClaimID']}' AND cr.ClaimSubjectID = cs.ID AND cs.CharID = '$char_id'", FALSE );

		// wrap it up
		$output = array( 'CanEnter' => FALSE, 'CanBuild' => FALSE, 'CanClaim' => FALSE, 'CanUse' => FALSE, 'CanDestroy' => FALSE );
		foreach( $permissions AS $rs ) {
			if( ! is_array($rs) ) continue;
			foreach( $rs AS $key => $value ) {
				$output[$key] = (bool)intval($value) ? TRUE : $output[$key];
			}
		}
		
		return $output;
		
	}
	
	# =============================================================================================================================================
	
	// Read the server info from SourceQuery API
	protected function query_server() {
		
		// Skip if private server
		if( ! $this->ispublic ) return FALSE;
		// Skip if we have the info already
		if( $this->serverinfo !== NULL ) return TRUE;
		
		// Connect to SQ if necessary
		if( ! $this->sq ) $this->sq_connect();
		
		// Fetch the information array from SourceQuery API
		try {
			$this->serverinfo = $this->sq->GetInfo();
			if( isSet($this->serverinfo['Players']) ) return TRUE;
			else return FALSE;
		// Handle error if no communication with LiF server possible
		} catch( Exception $e ) {
			return FALSE;
		}
		
	}

}

?>